/*
 * ECE3884 Lab01
 * Georgia Institute of Technology
 * Your Name: 
*/



/* ----------------------------------------------------------------------------
 *         ATMEL Microcontroller Software Support 
 * ----------------------------------------------------------------------------
 * Copyright (c) 2008, Atmel Corporation
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Atmel's name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ----------------------------------------------------------------------------
 */

//------------------------------------------------------------------------------
//         Headers
//------------------------------------------------------------------------------

#include <board.h>
#include <aic/aic.h>
#include <pit/pit.h>
#include <slcdc/slcdc.h>
#include <supc/supc.h>
#include <dbgu/dbgu.h>
#include <tc/tc.h>
#include <utility/trace.h>
#include <utility/assert.h>
#include <slcd/s7lstklcd/s7lstklcd.h>

#include <stdio.h>

//------------------------------------------------------------------------------
//         Local definitions
//------------------------------------------------------------------------------

/// Scrolling frequency in Hz.
#define SCROLLING_FREQUENCY     8

/// String is scrolling to the left
#define SCROLL_LEFT             0
/// String is scrolling to the right
#define SCROLL_RIGHT            1

//------------------------------------------------------------------------------
//         Local variables
//------------------------------------------------------------------------------

/// Pins to configure for the application.
static const Pin pPins[] = {PINS_DBGU};

/// String to display.
static const char pString[] = "ECE3884 Lab01";
/// Width in pixels of string to display.
static signed int stringWidth;

/// Scroll direction.
static unsigned char scrollDirection = SCROLL_LEFT;

//------------------------------------------------------------------------------
//         Local functions
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/// Initializes the SLCD controller with the board parameters.
//------------------------------------------------------------------------------
static void InitializeSlcd(void)
{
    // Enable SLCD internal power supply (3V)
    SUPC_EnableSlcd(AT91C_SUPC_LCDMODE_INTERNAL);
    SUPC_SetSlcdVoltage(0x9);
    
    // Disable SLCD
    SLCDC_Disable();
    
    //  Define the number of COM and SEG, Buffer driving time; select the bias
    SLCDC_Configure(S7LSTKLCD_NUM_COMMONS,
                    S7LSTKLCD_NUM_SEGMENTS,
                    AT91C_SLCDC_BIAS_1_3,
                    AT91C_SLCDC_BUFTIME_0_percent);
    
    // Set frame rate 30 Hz
    SLCDC_SetFrameFreq(AT91C_SLCDC_PRESC_SCLK_32, AT91C_SLCDC_DIV_1);
    
    //  Enable SLCD
    SLCDC_Enable();
}

//------------------------------------------------------------------------------
/// Interrupt service routine for the Timer Counter 0. Makes the displayed
/// string scroll from left to right.
//------------------------------------------------------------------------------
static void ISR_Tc0(void)
{
    static signed int offset = 0;

    AT91C_BASE_TC0->TC_SR;
    
    // Display string
    SLCDC_SetDisplayMode(6);
    SLCDC_Clear();
    S7LSTKLCD_String(offset, 1, pString);
    SLCDC_SetDisplayMode(AT91C_SLCDC_DISPMODE_NORMAL);

    // Scroll
    switch (scrollDirection) {

        case SCROLL_LEFT:
            if ((offset + stringWidth) <= S7LSTKLCD_WIDTH) {

                scrollDirection = SCROLL_RIGHT;
            }
            else {

                offset--;
            }
            break;

        case SCROLL_RIGHT:
            if (offset >= 0) {

                scrollDirection = SCROLL_LEFT;
            }
            else {

                offset++;
            }
            break;
    }
}

//------------------------------------------------------------------------------
//         Global functions
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
/// Main function
//------------------------------------------------------------------------------
int main(void)
{   
    unsigned int div, tcclks;

    // Configure DBGU
    PIO_Configure(pPins, PIO_LISTSIZE(pPins));
    DBGU_Configure(DBGU_STANDARD, 115200, BOARD_MCK);
    printf("\n\r-- ECE3884 Lab01 --\n\r");

    // Initialize SLCD
    InitializeSlcd();
    SLCDC_Clear();
    S7LSTKLCD_GetStringSize(pString, &stringWidth, 0);

    // Configure timer for interrupt every 250ms
    AT91C_BASE_PMC->PMC_PCER = 1 << AT91C_ID_TC0;
    TC_FindMckDivisor(SCROLLING_FREQUENCY, BOARD_MCK, &div, &tcclks);
    TC_Configure(AT91C_BASE_TC0, tcclks | AT91C_TC_CPCTRG);
    AT91C_BASE_TC0->TC_RC = (BOARD_MCK / div) / SCROLLING_FREQUENCY;
    AIC_ConfigureIT(AT91C_ID_TC0, 0, ISR_Tc0);
    AIC_EnableIT(AT91C_ID_TC0);
    AT91C_BASE_TC0->TC_IER = AT91C_TC_CPCS;
    TC_Start(AT91C_BASE_TC0);

    // Infinite loop
    while(1);
}
